/**
 *   Simulating Random Variations
 *   HTML5/Typescript version: 19 Aug 2017
 *   Original Flash version: Bowland Maths 2008
 *   Released under the Creative Commons attribution non-commercial, share alike 4.0 international licence
 *
 *   @version 20170819
 *   @author Daniel Pead
 *   @copyright © Bowland Charitable Trust 2008, © Shell Centre for Mathematical Education 2017
 *   @license CC BY-NC-SA 4.0
 *
 */
/**
 * The BarGraph view/renderer class
 */
var GraphView = (function () {
    function GraphView() {
        this.mLeftMargin = 80;
        this.mRightMargin = 20;
        this.mTopMargin = 40;
        this.mBottomMargin = 60;
        this.mGraphWidth = 600;
        this.mGraphHeight = 400;
        this.mXAxisLabel = "X Axis";
        this.mXIsCategoryAxis = true;
        this.mYAxisLabel = "Y Axis";
        this.mColourFromLabel = true;
        this.mTextPad = 10;
        this.mBarWidth = 0.8;
    }
    GraphView.prototype.render = function (svg) {
        console.log(svg.width(), svg.height());
        this.mGraphWidth = svg.width() - this.mRightMargin - this.mLeftMargin;
        this.mGraphHeight = svg.height() - this.mTopMargin - this.mBottomMargin;
        if (this.mGraph) {
            this.mGraph.remove();
        }
        this.mGraph = svg.group().id('graph');
        this.mGraph.translate(this.mLeftMargin, this.mTopMargin);
        this.mGraph.rect(this.mGraphWidth, this.mGraphHeight).addClass("graphDataFrame");
        this.mXLabels = this.mXAxisRange.getLabels(false);
        if (this.mXIsCategoryAxis) {
            this.mXStep = this.mXLabels.length > 0 ? this.mGraphWidth / (this.mXLabels.length) : this.mGraphWidth;
        }
        else {
            this.mXStep = this.mXLabels.length > 0 ? this.mGraphWidth / (this.mXLabels.length - 1) : this.mGraphWidth;
        }
        this.mYLabels = this.mYAxisRange.getLabels(false);
        this.mYStep = this.mYLabels.length > 1 ? this.mGraphHeight / (this.mYLabels.length - 1) : this.mGraphHeight;
        this.renderXAxis(this.mGraph);
        this.renderYAxis(this.mGraph);
        this.renderData(this.mGraph);
    };
    GraphView.prototype.renderXAxis = function (svg) {
        var g = svg.group().attr('id', 'grphXAxis');
        var svLabel = g.plain(this.mXAxisLabel).toggleClass("axisLabel").center(this.mGraphWidth / 2, this.mGraphHeight + this.mBottomMargin * 0.6);
        var y = this.mGraphHeight;
        var sg = g.group().id('catAxisScale');
        var gg = g.group().id('catAxisGrid');
        var nl = this.mXLabels.length;
        var th = 0;
        var xoff = 0;
        if (this.mXIsCategoryAxis) {
            xoff = this.mXStep / 2;
        }
        for (var i = 0; i < nl; i++) {
            var x = i * this.mXStep;
            if (i > 0)
                gg.line(x, 0, x, this.mGraphHeight);
            x += xoff;
            var t = sg.plain(this.mXLabels[i]).addClass("axisScale").center(x, y);
            if (th == 0)
                th = t.bbox().height;
            t.dy(th + this.mTextPad);
        }
    };
    GraphView.prototype.renderYAxis = function (svg) {
        var g = svg.group().id('grphYAxis');
        var svLabel = g.plain(this.mYAxisLabel).toggleClass("axisLabel").center(-this.mLeftMargin * 0.8, this.mGraphHeight * 0.5);
        svLabel.rotate(-90);
        var sg = g.group().id('valAxisScale');
        var gg = g.group().id('valAxisGrid');
        var nl = this.mYLabels.length;
        for (var i = 0; i < nl; i++) {
            var y = this.mGraphHeight - i * this.mYStep;
            if (i > 0 && i + 1 < nl)
                gg.line(0, y, this.mGraphWidth, y);
            var t = sg.plain(this.mYLabels[i]).cy(y);
            t.dx(-t.bbox().width - this.mTextPad);
        }
    };
    GraphView.prototype.renderData = function (svg) {
        var nl = Math.min(this.mXLabels.length, this.mData.length);
        var w = this.mXStep * this.mBarWidth;
        var xo = (this.mXStep - w) / 2;
        var yscale = this.mGraphHeight / this.mYAxisRange.max;
        var g = svg.group().id('dataBars');
        for (var i = 0; i < nl; i++) {
            var c = parseInt(this.mXLabels[i]);
            if (isNaN(c) || !this.mColourFromLabel)
                c = i + 1;
            else
                c = ((c - 1) % 20) + 1;
            var h = this.mData[i] * yscale;
            g.rect(w, h).move(this.mXStep * i + xo, this.mGraphHeight - h).addClass('d_' + c);
        }
    };
    return GraphView;
}());
/**
 *   Simulating Random Variations
 *   HTML5/Typescript version: 19 Aug 2017
 *   Original Flash version: Bowland Maths 2008
 *   Released under the Creative Commons attribution non-commercial, share alike 4.0 international licence
 *
 *   @version 20170819
 *   @author Daniel Pead
 *   @copyright © Bowland Charitable Trust 2008, © Shell Centre for Mathematical Education 2017
 *   @license CC BY-NC-SA 4.0
 */
/**
 * NumberRange Utility class
 * used by VarSim simulation class and for axis scaling and labelling.
 */
var NumberRange = (function () {
    function NumberRange(min, max, step, places) {
        if (max === void 0) { max = NaN; }
        if (step === void 0) { step = NaN; }
        if (places === void 0) { places = 0; }
        this.mPlaces = 0;
        this.mMin = this.isNaN(max) ? min : Math.min(min, max);
        this.mMax = this.isNaN(min) ? max : Math.max(min, max);
        this.mStep = this.isNaN(step) ? NaN : step;
        this.mPlaces = this.isNaN(places) ? 0 : places;
    }
    /**
     * Extend the range to include number n
     * @param n Number to include
     */
    NumberRange.prototype.includeValue = function (n) {
        if (this.isNaN(n))
            return;
        this.mMax = this.isNaN(this.mMax) ? n : Math.max(n, this.mMax);
        this.mMin = this.isNaN(this.mMin) ? n : Math.min(n, this.mMin);
        this.mStep = null;
    };
    /**
     * Works more like ActionScript isNaN - null is not a number!
     * @param n Number to test
     */
    NumberRange.prototype.isNaN = function (n) {
        return isNaN(parseFloat(n)) || !isFinite(n);
    };
    /**
     * Return new NumberRange with suggested step size and max/min
     * adjusted to multiples of the step size
     */
    NumberRange.prototype.makeScale = function (steps, integer) {
        console.log(this.mMin + "-" + this.mMax);
        if ((this.mMax - this.mMin) <= Number.MIN_VALUE) {
            // Sensible failsafe
            return new NumberRange(0, this.isNaN(steps) ? 10 : steps, 1, 0);
        }
        var step = this.mStep;
        var places = this.mPlaces;
        if (!(this.isNaN(step) || this.isNaN(steps))) {
            var currSteps = ((this.mMax - this.mMin) / step) / steps;
            if (currSteps < 0.8 || currSteps > 1.2) {
                step = null;
            }
        }
        if (steps == null)
            steps = 10;
        if (this.isNaN(step)) {
            step = Math.log((this.mMax - this.mMin) / steps) / Math.LN10;
            console.log(step);
            var power = Math.floor(step);
            step -= power;
            if (step <= 0.3)
                step = Math.pow(10, power);
            else if (step < 0.69)
                step = 2 * Math.pow(10, power);
            else
                step = 5 * Math.pow(10, power);
            if (step < 1) {
                if (integer == true)
                    step = 1;
                else
                    places = 1 + Math.abs(power);
            }
        }
        var min = step * Math.floor(this.mMin / step);
        var max = step * Math.ceil(this.mMax / step);
        console.log(min + "-" + max + " x " + step);
        return new NumberRange(min, max, step, places);
    };
    NumberRange.prototype.getLabels = function (catdata) {
        var l = [];
        if (this.isNaN(this.mMin) || this.isNaN(this.mMax) || !(this.mStep > 0))
            return l;
        for (var n = this.mMin; n <= this.mMax; n += this.mStep) {
            if (catdata)
                l.push(n + "-" + (n + this.mStep));
            else
                l.push(n.toString());
            //console.log ( l[l.length-1] );
        }
        return l;
    };
    NumberRange.prototype.numSteps = function () {
        return Math.ceil((this.mMax - this.mMin) / this.mStep);
    };
    NumberRange.prototype.getBinNumber = function (n) {
        console.log(this.mMin + " " + this.mMax + " " + this.mStep);
        if (n < this.mMin || n >= this.mMax)
            return NaN;
        return Math.floor((n - this.mMin) / this.mStep);
    };
    NumberRange.prototype.toString = function () {
        return "{" + this.mMin + "-" + this.mMax + " this.mStep " + this.mStep + " to " + this.mPlaces + " dp}";
    };
    NumberRange.prototype.clone = function () {
        return new NumberRange(this.mMin, this.mMax, this.mStep, this.mPlaces);
    };
    NumberRange.prototype.shift = function (steps) {
        this.mMax += steps * this.mStep;
        this.mMin += steps * this.mStep;
    };
    Object.defineProperty(NumberRange.prototype, "min", {
        get: function () { return this.mMin; },
        enumerable: true,
        configurable: true
    });
    Object.defineProperty(NumberRange.prototype, "max", {
        get: function () { return this.mMax; },
        enumerable: true,
        configurable: true
    });
    Object.defineProperty(NumberRange.prototype, "step", {
        get: function () { return this.mStep; },
        enumerable: true,
        configurable: true
    });
    Object.defineProperty(NumberRange.prototype, "places", {
        get: function () { return this.mPlaces; },
        enumerable: true,
        configurable: true
    });
    return NumberRange;
}());
/**
 *   Simulating Random Variations
 *   HTML5/Typescript version: 19 Aug 2017
 *   Original Flash version: Bowland Maths 2008
 *   Released under the Creative Commons attribution non-commercial, share alike 4.0 international licence
 *
 *   @version 20170819
 *   @author Daniel Pead
 *   @copyright © Bowland Charitable Trust 2008, © Shell Centre for Mathematical Education 2017
 *   @license CC BY-NC-SA 4.0
 *
 *   jQuery and svg.js libraries are included under the MIT license - see relavant source files for details.
 */
$(function () {
    Risk.getInstance().start();
});
/**
 * Main applet singleton class
 */
var Risk = (function () {
    function Risk() {
        /** True if the frequency graph is showing */
        this.showingFreq = true;
    }
    Risk.prototype.start = function () {
        var _this = this;
        console.log("Hello World 2");
        this.svg = SVG('graph').size($('#graph').width(), $('#graph').height());
        SVG.defaults.attrs['font-family'] = null;
        this.sim = new VarSim();
        this.graph = new GraphView();
        this.showCasesGraph();
        this.showingFreq = false;
        $('#btnNextTrial').on('click', function () { _this.nextTrial(1); });
        $('#btn20Years').on('click', function () { _this.nextTrial(20); });
        $('#btnResetSim').on('click', function () { _this.resetSim(); });
        $('#btnFreqGraph').on('click', function () {
            _this.showFrequencyGraph();
        });
        $('#btnCasesGraph').on('click', function () {
            _this.showCasesGraph();
        });
        $('#population').on('input', function () {
            _this.updateProbability();
        });
        $('#deaths').on('input', function () {
            _this.updateProbability();
        });
        $('#toggleHigh').on('click', function () {
            $('body').toggleClass("high").toggleClass("low", false);
            return false;
        });
        $('#toggleLow').on('click', function () {
            $('body').toggleClass("low").toggleClass("high", false);
            return false;
        });
        $('#help').hide();
        $('#btnHelp').on('click', function () {
            _this.toggleHelp();
        });
        $(document).on("keydown", function (e) {
            console.log(e.keyCode);
            switch (e.keyCode) {
                case 78:
                    _this.nextTrial(1);
                    return false; // n
                case 83:
                    _this.resetSim();
                    return false; // s
                case 84:
                    _this.nextTrial(20);
                    return false; // t
                case 70:
                    _this.showFrequencyGraph();
                    return false; // f
                case 89:
                    _this.showCasesGraph();
                    return false; // y
                case 80:
                    $('#population').focus().select();
                    return false; // p
                case 68:
                    $('#deaths').focus().select();
                    return false; // c
                case 76:
                    $('body').toggleClass("low").toggleClass("high", false);
                    return false; //l
                case 72:
                    $('body').toggleClass("high").toggleClass("low", false);
                    return false; //h
                case 191:
                    _this.toggleHelp();
                    return false; //?
            }
            return true;
        });
    };
    Risk.prototype.toggleHelp = function () {
        var jqHelp = $("#help");
        if (jqHelp.is(':visible')) {
            jqHelp.hide();
            $('#btnHelp').text("?");
        }
        else {
            jqHelp.show();
            $('#btnHelp').text("X");
        }
    };
    Risk.prototype.updateProbability = function () {
        var pop = parseInt($('#population').val());
        var deaths = parseInt($('#deaths').val());
        var oldPop = this.sim.getPopulation();
        var oldDeaths = Math.round(oldPop * this.sim.getProbability());
        var tTrials = this.sim.getTotTrials();
        var nogood = isNaN(pop) || pop <= 0 || pop > VarSim.MAXPOP || isNaN(deaths) || deaths < 0 || deaths > pop;
        console.log(pop, deaths, nogood);
        if (pop < 0)
            $('#population').val(oldPop.toString());
        if (deaths < 0)
            $('#deaths').val(oldDeaths.toString());
        if (isNaN(pop) || pop <= 0)
            pop = oldPop;
        if (isNaN(deaths) || deaths < 0)
            deaths = 0;
        if (deaths > pop)
            deaths = pop;
        this.sim.setByCases(pop, deaths);
        console.log("Changed " + this.sim.getProbability());
        var pr = this.sim.getProbRecip();
        if (isFinite(pr))
            $('#probability').val("1 in " + pr);
        else
            $('#probability').val("0");
        this.sim.resetSim();
        if (tTrials > 1)
            this.sim.runTrials(tTrials - 1);
        if (this.showingFreq) {
            this.showFrequencyGraph();
        }
        else {
            this.showCasesGraph();
        }
        return true;
    };
    Risk.prototype.showCasesGraph = function () {
        this.graph.mXAxisRange = this.sim.getYearRange();
        this.graph.mYAxisRange = this.sim.getCasesByYearRange();
        this.graph.mData = this.sim.getCasesByYear();
        this.graph.mXIsCategoryAxis = true;
        this.graph.mColourFromLabel = true;
        this.graph.mXAxisLabel = "Number of deaths";
        this.graph.mYAxisLabel = "Year";
        this.graph.render(this.svg);
        this.showingFreq = false;
        this.showFigures();
        $('#btnFreqGraph').show();
        $('#btnCasesGraph').hide();
    };
    Risk.prototype.showFrequencyGraph = function () {
        this.graph.mXAxisRange = this.sim.getFreqBinRange();
        this.graph.mYAxisRange = this.sim.getFreqRange();
        this.graph.mData = this.sim.getFreq();
        console.log(this.graph.mData);
        this.graph.mXIsCategoryAxis = this.graph.mXAxisRange.step == 1;
        this.graph.mColourFromLabel = false;
        this.graph.mXAxisLabel = "Deaths in 1 year";
        this.graph.mYAxisLabel = "Frequency";
        this.graph.render(this.svg);
        this.showingFreq = true;
        this.showFigures();
        $('#btnFreqGraph').hide();
        $('#btnCasesGraph').show();
    };
    Risk.prototype.showFigures = function () {
        $('#outYears').val(this.sim.getTotTrials().toString());
        $('#outLast').val(this.sim.getLastCases().toString());
    };
    Risk.prototype.nextTrial = function (n) {
        var t = this.sim.getTotTrials();
        if (n > 1 && t < n)
            n = n - t;
        if (t + n > 9000)
            n = 9000 - t;
        if (n) {
            this.sim.runTrials(n);
            if (this.showingFreq) {
                this.showFrequencyGraph();
            }
            else {
                this.showCasesGraph();
            }
        }
        t += n;
        if (t >= 9000) {
            $('#btnNextTrial').attr("disabled", "disabled");
        }
        if (t + 20 > 9000) {
            $('#btn20Years').attr("disabled", "disabled");
        }
    };
    Risk.prototype.resetSim = function () {
        this.sim.resetSim();
        this.showCasesGraph();
        $('#btnNextTrial').removeAttr("disabled");
        $('#btn20Years').removeAttr("disabled");
    };
    Risk.getInstance = function () {
        if (!Risk.instance)
            Risk.instance = new Risk();
        return Risk.instance;
    };
    return Risk;
}());
/**
 *   Simulating Random Variations
 *   HTML5/Typescript version: 19 Aug 2017
 *   Original Flash version: Bowland Maths 2008
 *   Released under the Creative Commons attribution non-commercial, share alike 4.0 international licence
 *
 *   @version 20170819
 *   @author Daniel Pead
 *   @copyright © Bowland Charitable Trust 2008, © Shell Centre for Mathematical Education 2017
 *   @license CC BY-NC-SA 4.0
 */
/**
 * The main model class
 * Simulates the number of low-probability events expected in a population over a number of years
 */
var VarSim = (function () {
    function VarSim(numTrials) {
        if (numTrials === void 0) { numTrials = 0; }
        this.aNormal = [];
        this.setByCases(VarSim.POP, VarSim.CASES);
        if (!numTrials)
            numTrials = VarSim.NUMYEARS;
        this.resetSim(numTrials);
    }
    VarSim.prototype.rand = function () {
        return Math.random();
    };
    /**
     * Generate4s a pair of pseudo-random values with normal distribution, mean=0 and variance=1
     * Uses the Box-Muller method (thanks to "Numerical Recipies")
     * Values stored in this.aNormal
     * use getNormal() to retrieve values one at a time
     */
    VarSim.prototype.genNormalPair = function () {
        var v1;
        var v2;
        var r;
        var f;
        // To quote Linus Torvalds: "You are not supposed to understand this".
        v1 = 2 * this.rand() - 1;
        do {
            v2 = 2 * this.rand() - 1;
            r = (v1 * v1) + (v2 * v2);
        } while (r > 1);
        f = Math.sqrt(-2.0 * Math.log(r) / r);
        this.aNormal[0] = v1 * f;
        this.aNormal[1] = v2 * f;
    };
    /**
     *  Returns pseudo-random values with normal distribution, mean=0 and variance=1
     */
    VarSim.prototype.getNormal = function () {
        if (!this.aNormal.length) {
            this.genNormalPair();
        }
        return this.aNormal.pop();
    };
    VarSim.prototype.resetSim = function (trials) {
        if (trials === void 0) { trials = NaN; }
        if (isNaN(trials))
            trials = VarSim.NUMYEARS;
        this.maxTrials = trials;
        this.totTrials = 1; //trials;
        this.trialIndex = 1;
        this.sd = Math.sqrt(this.population * this.probability * (1 - this.probability));
        // Range for frequency table
        var f0 = Math.max(Math.floor(this.cases - 2.5 * this.sd), 0);
        var f1 = Math.ceil(Math.floor(this.cases + 2.5 * this.sd + 1));
        if (f0 < 0)
            f0 = 0;
        if ((f1 - f0) < 10) {
            f0 = 0;
            f1 = 10;
        }
        this.freqBinRange = new NumberRange(f0, f1);
        this.freqBinRange = this.freqBinRange.makeScale(10, true);
        this.freq = new Array(this.freqBinRange.numSteps());
        for (var i = 0; i < this.freq.length; this.freq[i++] = NaN)
            ;
        var bin = this.freqBinRange.getBinNumber(this.cases);
        if (!isNaN(bin))
            this.freq[bin] = 1;
        this.casesByYear = new Array(trials);
        this.casesByYearRange = new NumberRange(0);
        this.casesByYearRange.includeValue(0);
        this.casesByYear[0] = this.cases;
        this.casesByYearRange.includeValue(this.cases);
        this.casesByYearRange.includeValue(Math.ceil(this.cases + 4 * this.sd));
        this.lastCases = this.cases;
        this.freqRange = new NumberRange(1, 10, 1);
        this.yearRange = new NumberRange(1, this.maxTrials, 1);
    };
    VarSim.prototype.runOnce = function () {
        var n;
        if (this.cases > this.population || this.population == 0)
            return 0;
        do {
            n = Math.round(this.cases + this.sd * this.getNormal());
        } while (n < 0 || n > this.population);
        if (this.totTrials >= this.maxTrials) {
            this.casesByYear.shift();
            //this.totTrials--;
            this.trialIndex = this.maxTrials - 1;
            this.yearRange.shift(1);
        }
        this.casesByYear[this.trialIndex] = n;
        this.casesByYearRange.includeValue(n);
        var bin = this.freqBinRange.getBinNumber(n);
        if ((!isNaN(bin)) && bin < this.freq.length) {
            if (isNaN(this.freq[bin]))
                this.freq[bin] = 0;
            this.freq[bin]++;
            this.freqRange.includeValue(this.freq[bin] + 1);
        }
        this.lastCases = n;
        this.totTrials++;
        this.trialIndex++;
        return n;
    };
    VarSim.prototype.runTrials = function (t) {
        while (t-- > 0) {
            this.runOnce();
        }
    };
    VarSim.prototype.dumpFreq = function () {
        var fl = this.freqBinRange.getLabels(false);
        console.log("fl " + fl);
        for (var i = 0; i < fl.length; i++) {
            console.log("Cases " + fl[i] + " freq " + this.freq[i]);
        }
    };
    VarSim.prototype.getCasesByYear = function () {
        return this.casesByYear;
    };
    VarSim.prototype.setByProb = function (population, prob) {
        this.population = population;
        this.probability = prob;
        this.cases = population * prob;
    };
    VarSim.prototype.setByCases = function (population, cases) {
        this.population = population;
        this.cases = cases;
        this.probability = cases / population;
    };
    VarSim.prototype.getTotTrials = function () {
        return this.totTrials;
    };
    VarSim.prototype.getMaxTrials = function () {
        return this.maxTrials;
    };
    VarSim.prototype.getLastCases = function () {
        return this.lastCases;
    };
    VarSim.prototype.getProbability = function () {
        return this.probability;
    };
    VarSim.prototype.getSd = function () {
        return this.sd;
    };
    VarSim.prototype.getCases = function () {
        return Math.round(this.cases);
    };
    VarSim.prototype.getPopulation = function () {
        return Math.round(this.population);
    };
    VarSim.prototype.getCasesByYearRange = function () {
        var r = this.casesByYearRange.makeScale(10, true);
        if (r.max < 10)
            r = new NumberRange(0, 10, 1);
        return r;
    };
    VarSim.prototype.getYearRange = function () {
        return this.yearRange;
    };
    VarSim.prototype.getFreqBinRange = function () {
        return this.freqBinRange;
    };
    VarSim.prototype.getFreq = function () {
        return this.freq;
    };
    VarSim.prototype.getFreqRange = function () {
        return this.freqRange.makeScale(10, true);
    };
    VarSim.prototype.getProbRecip = function () {
        return Math.round(1 / this.probability);
    };
    VarSim.NUMYEARS = 20;
    VarSim.POP = 50000000;
    VarSim.MAXPOP = 500000000;
    VarSim.CASES = 1000;
    return VarSim;
}());
//# sourceMappingURL=risk.js.map